package tcc.haruki.domain;

import java.math.BigInteger;
import java.util.Arrays;

/**
 * Representa um elemento de GF(p<sup>2</sup>).
 * @author haruki
 *
 */
public class XTRElem {
	private GaloisField gf;
	
	private BigInteger[] coef;
	
	public XTRElem (GaloisField gf, BigInteger b0, BigInteger b1) {
		this.gf = gf;
		this.coef = new BigInteger[2];
		this.coef[0] = gf.val(b0);
		this.coef[1] = gf.val(b1);
	}
	
	public XTRElem (GaloisField gf, BigInteger[] coef) {
		this(gf, coef[0], coef[1]);
	}
	
	public XTRElem expP() {
		return new XTRElem(this.gf, this.coef[1], this.coef[0]);
	}
	
	public XTRElem exp2() {
		BigInteger dois = new BigInteger("2");
		BigInteger e1 = coef[1].multiply(coef[1].subtract(coef[0].multiply(dois)));
		BigInteger e2 = coef[0].multiply(coef[0].subtract(coef[1].multiply(dois)));
		
		return new XTRElem(this.gf, e1, e2);
	}
	
	public XTRElem add(XTRElem elem) {
		BigInteger e1 = coef[0].add(elem.coef[0]);
		BigInteger e2 = coef[1].add(elem.coef[1]);
		
		return new XTRElem(this.gf, e1, e2);
	}
	
	public XTRElem sub(XTRElem elem) {
		BigInteger e1 = coef[0].subtract(elem.coef[0]);
		BigInteger e2 = coef[1].subtract(elem.coef[1]);
		
		return new XTRElem(this.gf, e1, e2);
	}
	
	public XTRElem mul(XTRElem elem) {
		BigInteger x1y1 = coef[0].multiply(elem.coef[0]);
		BigInteger x2y2 = coef[1].multiply(elem.coef[1]);
		BigInteger sumCoefs = coef[0].add(coef[1]);
		BigInteger elemSumCoefs = elem.coef[0].add(elem.coef[1]);
		BigInteger temp = sumCoefs.multiply(elemSumCoefs);
		BigInteger aux = temp.subtract(x1y1).subtract(x2y2);
		BigInteger e1 = x2y2.subtract(aux);
		BigInteger e2 = x1y1.subtract(aux);
		
		return new XTRElem(this.gf, e1, e2);
	}
	
	public XTRElem scalarMul(BigInteger scalar) {
		BigInteger e1 = coef[0].multiply(scalar);
		BigInteger e2 = coef[1].multiply(scalar);
		return new XTRElem(this.gf, e1, e2);
	}
	
	
	public XTRElem mulInv(XTRElem elem) {
		return this.mul(new XTRElem(this.gf, this.gf.inverse(elem.coef[0]), this.gf.inverse(elem.coef[1])));
	}
	
	public boolean isGFElement() {
		if (this.coef[0].equals(this.coef[1]))
			return true;
		return false;
	}
	
	public BigInteger getGFvalue() {
		if (!isGFElement()) {
			System.err.println("ALO ERRO");
		}
		return gf.val(this.coef[0].negate());
	}
	
	public byte[] getByteRepresentation() {
		byte[] rep1 = coef[0].toByteArray();
		byte[] rep2 = coef[1].toByteArray();
		
		int size1 = rep1.length;
		int size2 = rep2.length;
		
		byte[] size1b = new BigInteger(Integer.toString(size1)).toByteArray();
		byte[] size2b = new BigInteger(Integer.toString(size2)).toByteArray();
		
		byte[] ret = new byte[rep1.length + rep2.length + 8];
		
		for (int i = 0; i < 8; i++) {
			ret[i] = 0;
		}
		
		int idx = 3;
		for (int i = size1b.length-1; i >= 0; i--) {
			ret[idx--] = size1b[i];
		}
		
		idx = 7;
		for (int i = size2b.length-1; i >= 0; i--) {
			ret[idx--] = size2b[i];
		}
	
		idx = 8;
		for (int i = 0; i < rep1.length; i++) {
			ret[idx++] = rep1[i];
		}
		for (int i = 0; i < rep2.length; i++) {
			ret[idx++] = rep2[i];
		}
		
		return ret;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + Arrays.hashCode(coef);
		result = prime * result + ((gf == null) ? 0 : gf.hashCode());
		return result;
	}

	
}
